<?php

namespace App\Livewire\Reports;

use Livewire\Component;
use Livewire\Attributes\Title;
use App\Models\Translation;
use App\Models\Order;
use App\Models\Payment;
use App\Models\Expense;
use Carbon\Carbon;

class MonthlySummary extends Component
{
    #[Title('Monthly Summary')]
    
    public $month;
    public $year;
    public $lang;

    // Report Data Properties
    public $total_revenue = 0;
    public $total_orders = 0;
    public $total_gross_income = 0;
    public $total_net_income = 0;
    public $total_expense = 0;
    public $total_tax = 0;

    public function mount()
    {
        // Permission Check
        if(!\Illuminate\Support\Facades\Gate::allows('report_monthly_summary')){
            abort(404);
        }

        // Default to current month/year
        $this->month = Carbon::now()->month;
        $this->year = Carbon::now()->year;

        if (session()->has('selected_language')) {
            $this->lang = Translation::where('id', session()->get('selected_language'))->first();
        } else {
            $this->lang = Translation::where('default', 1)->first();
        }
        
        $this->generateReport();
    }

    public function updated($name, $value)
    {
        // Regenerate report when month or year changes
        if ($name == 'month' || $name == 'year') {
            $this->generateReport();
        }
    }

    public function generateReport()
    {
        // Ensure month and year are valid
        if (!$this->month || !$this->year) {
            return;
        }

        // 1. Total Orders (Month)
        // Using 'order_date' as standard
        $this->total_orders = Order::whereMonth('order_date', $this->month)
                                   ->whereYear('order_date', $this->year)
                                   ->count();

        // 2. Total Gross Income (Revenue) 
        // Based on Payments received in that month
        $this->total_gross_income = Payment::whereMonth('payment_date', $this->month)
                                           ->whereYear('payment_date', $this->year)
                                           ->sum('received_amount');
        
        // Use consistent variable naming if 'total_revenue' is meant to be same as 'gross_income'
        $this->total_revenue = $this->total_gross_income;

        // 3. Total Expense (Month)
        $this->total_expense = Expense::whereMonth('expense_date', $this->month)
                                      ->whereYear('expense_date', $this->year)
                                      ->sum('expense_amount');

        // 4. Total Tax (Month) - Only from Delivered Orderss
        $this->total_tax = Order::whereMonth('order_date', $this->month)
                                ->whereYear('order_date', $this->year)
                                ->where('status', 3) // Delivered
                                ->sum('tax_amount');

        // 5. Total Net Income
        $this->total_net_income = $this->total_gross_income - $this->total_expense - $this->total_tax;
    }

    public function render()
    {
        return view('livewire.reports.monthly-summary');
    }
}
